------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               S N A M E S                                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--          Copyright (C) 1992-2021, Free Software Foundation, Inc.         --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT; see file COPYING3.  If not, go to --
-- http://www.gnu.org/licenses for a complete copy of the license.          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

with Debug; use Debug;
with Opt;   use Opt;
with Table;
with Types; use Types;

package body Snames is

   --  Table used to record convention identifiers

   type Convention_Id_Entry is record
      Name       : Name_Id;
      Convention : Convention_Id;
   end record;

   package Convention_Identifiers is new Table.Table (
     Table_Component_Type => Convention_Id_Entry,
     Table_Index_Type     => Int,
     Table_Low_Bound      => 1,
     Table_Initial        => 50,
     Table_Increment      => 200,
     Table_Name           => "Name_Convention_Identifiers");

   --  Table of names to be set by Initialize. Each name is terminated by a
   --  single #, and the end of the list is marked by a null entry, i.e. by
   --  two # marks in succession. Note that the table does not include the
   --  entries for a-z, since these are initialized by Namet itself.

   Preset_Names : constant String :=
     "_parent#" &
     "_tag#" &
     "off#" &
     "space#" &
     "time#" &
     "default_value#" &
     "default_component_value#" &
     "designated_storage_model#" &
     "dimension#" &
     "dimension_system#" &
     "disable_controlled#" &
     "dynamic_predicate#" &
     "exclusive_functions#" &
     "full_access_only#" &
     "integer_literal#" &
     "no_controlled_parts#" &
     "no_task_parts#" &
     "real_literal#" &
     "relaxed_initialization#" &
     "stable_properties#" &
     "static_predicate#" &
     "storage_model#" &
     "storage_model_type#" &
     "string_literal#" &
     "synchronization#" &
     "unimplemented#" &
     "_abort_signal#" &
     "_alignment#" &
     "_assign#" &
     "_atcb#" &
     "_chain#" &
     "_controller#" &
     "_cpu#" &
     "_dispatching_domain#" &
     "_finalizer#" &
     "_finalizer_old#" &
     "_finalization_controller#" &
     "_init#" &
     "_init_level#" &
     "_invariant#" &
     "_master#" &
     "_object#" &
     "_post#" &
     "_postcond_enabled#" &
     "_pre#" &
     "_priority#" &
     "_relative_deadline#" &
     "_result#" &
     "_result_object_for_postcond#" &
     "_return_success_for_postcond#" &
     "_secondary_stack#" &
     "_secondary_stack_size#" &
     "_size#" &
     "_stack#" &
     "_task#" &
     "_task_id#" &
     "_task_info#" &
     "_task_name#" &
     "_type_invariant#" &
     "_variants#" &
     "_wrapped_statements#" &
     "_disp_asynchronous_select#" &
     "_disp_conditional_select#" &
     "_disp_get_prim_op_kind#" &
     "_disp_get_task_id#" &
     "_disp_requeue#" &
     "_disp_timed_select#" &
     "initialize#" &
     "adjust#" &
     "finalize#" &
     "finalize_address#" &
     "next#" &
     "prev#" &
     "allocate#" &
     "deallocate#" &
     "dereference#" &
     "decimal_io#" &
     "enumeration_io#" &
     "fixed_io#" &
     "float_io#" &
     "integer_io#" &
     "modular_io#" &
     "dim_symbol#" &
     "item#" &
     "put_dim_of#" &
     "sqrt#" &
     "symbol#" &
     "unit_symbol#" &
     "const#" &
     "error#" &
     "false#" &
     "go#" &
     "put#" &
     "put_line#" &
     "to#" &
     "defined#" &
     "arith_64#" &
     "arith_128#" &
     "exception_traces#" &
     "finalization#" &
     "interfaces#" &
     "memory#" &
     "memory_compare#" &
     "memory_copy#" &
     "memory_move#" &
     "memory_set#" &
     "most_recent_exception#" &
     "stack_checking#" &
     "standard#" &
     "system#" &
     "text_io#" &
     "wide_text_io#" &
     "wide_wide_text_io#" &
     "abort_task#" &
     "bounded_io#" &
     "c_streams#" &
     "complex_io#" &
     "directories#" &
     "direct_io#" &
     "dispatching#" &
     "editing#" &
     "edf#" &
     "reset_standard_files#" &
     "sequential_io#" &
     "strings#" &
     "streams#" &
     "suspend_until_true#" &
     "suspend_until_true_and_set_deadline#" &
     "synchronous_barriers#" &
     "task_identification#" &
     "text_streams#" &
     "unbounded#" &
     "unbounded_io#" &
     "wait_for_release#" &
     "wide_unbounded#" &
     "wide_wide_unbounded#" &
     "yield#" &
     "no_dsa#" &
     "garlic_dsa#" &
     "polyorb_dsa#" &
     "addr#" &
     "async#" &
     "get_active_partition_id#" &
     "get_rci_package_receiver#" &
     "get_rci_package_ref#" &
     "origin#" &
     "params#" &
     "partition#" &
     "ras#" &
     "_call#" &
     "rci_name#" &
     "receiver#" &
     "rpc#" &
     "subp_id#" &
     "operation#" &
     "argument#" &
     "arg_modes#" &
     "target#" &
     "req#" &
     "obj_typecode#" &
     "stub#" &
     "Oabs#" &
     "Oand#" &
     "Omod#" &
     "Onot#" &
     "Oor#" &
     "Orem#" &
     "Oxor#" &
     "Oeq#" &
     "One#" &
     "Olt#" &
     "Ole#" &
     "Ogt#" &
     "Oge#" &
     "Oadd#" &
     "Osubtract#" &
     "Oconcat#" &
     "Omultiply#" &
     "Odivide#" &
     "Oexpon#" &
     "ada_83#" &
     "ada_95#" &
     "ada_05#" &
     "ada_2005#" &
     "ada_12#" &
     "ada_2012#" &
     "ada_2022#" &
     "aggregate_individually_assign#" &
     "allow_integer_address#" &
     "annotate#" &
     "assertion_policy#" &
     "assume_no_invalid_values#" &
     "c_pass_by_copy#" &
     "check_float_overflow#" &
     "check_name#" &
     "check_policy#" &
     "component_alignment#" &
     "convention_identifier#" &
     "debug_policy#" &
     "default_storage_pool#" &
     "detect_blocking#" &
     "disable_atomic_synchronization#" &
     "discard_names#" &
     "elaboration_checks#" &
     "eliminate#" &
     "enable_atomic_synchronization#" &
     "extend_system#" &
     "extensions_allowed#" &
     "external_name_casing#" &
     "favor_top_level#" &
     "gnat_annotate#" &
     "ignore_pragma#" &
     "implicit_packing#" &
     "initialize_scalars#" &
     "interrupt_state#" &
     "license#" &
     "locking_policy#" &
     "no_component_reordering#" &
     "no_heap_finalization#" &
     "no_run_time#" &
     "no_strict_aliasing#" &
     "normalize_scalars#" &
     "optimize_alignment#" &
     "overflow_mode#" &
     "overriding_renamings#" &
     "partition_elaboration_policy#" &
     "persistent_bss#" &
     "prefix_exception_messages#" &
     "priority_specific_dispatching#" &
     "profile#" &
     "profile_warnings#" &
     "propagate_exceptions#" &
     "queuing_policy#" &
     "rational#" &
     "ravenscar#" &
     "rename_pragma#" &
     "restricted_run_time#" &
     "restrictions#" &
     "restriction_warnings#" &
     "reviewable#" &
     "short_circuit_and_or#" &
     "short_descriptors#" &
     "source_file_name#" &
     "source_file_name_project#" &
     "spark_mode#" &
     "style_checks#" &
     "suppress#" &
     "suppress_exception_locations#" &
     "task_dispatching_policy#" &
     "unevaluated_use_of_old#" &
     "unsuppress#" &
     "use_vads_size#" &
     "validity_checks#" &
     "warning_as_error#" &
     "warnings#" &
     "wide_character_encoding#" &
     "abort_defer#" &
     "abstract_state#" &
     "all_calls_remote#" &
     "assert#" &
     "assert_and_cut#" &
     "assume#" &
     "async_readers#" &
     "async_writers#" &
     "asynchronous#" &
     "atomic#" &
     "atomic_components#" &
     "attach_handler#" &
     "attribute_definition#" &
     "check#" &
     "comment#" &
     "common_object#" &
     "compile_time_error#" &
     "compile_time_warning#" &
     "complete_representation#" &
     "complex_representation#" &
     "constant_after_elaboration#" &
     "contract_cases#" &
     "controlled#" &
     "convention#" &
     "cpp_class#" &
     "cpp_constructor#" &
     "cpp_virtual#" &
     "cpp_vtable#" &
     "cuda_device#" &
     "cuda_execute#" &
     "cuda_global#" &
     "deadline_floor#" &
     "debug#" &
     "default_initial_condition#" &
     "depends#" &
     "effective_reads#" &
     "effective_writes#" &
     "elaborate#" &
     "elaborate_all#" &
     "elaborate_body#" &
     "export#" &
     "export_function#" &
     "export_object#" &
     "export_procedure#" &
     "export_valued_procedure#" &
     "extensions_visible#" &
     "external#" &
     "finalize_storage_only#" &
     "ghost#" &
     "global#" &
     "ident#" &
     "implementation_defined#" &
     "implemented#" &
     "import#" &
     "import_function#" &
     "import_object#" &
     "import_procedure#" &
     "import_valued_procedure#" &
     "independent#" &
     "independent_components#" &
     "initial_condition#" &
     "initializes#" &
     "inline#" &
     "inline_always#" &
     "inline_generic#" &
     "inspection_point#" &
     "interface_name#" &
     "interrupt_handler#" &
     "invariant#" &
     "keep_names#" &
     "link_with#" &
     "linker_alias#" &
     "linker_constructor#" &
     "linker_destructor#" &
     "linker_options#" &
     "linker_section#" &
     "list#" &
     "lock_free#" &
     "loop_invariant#" &
     "loop_optimize#" &
     "loop_variant#" &
     "machine_attribute#" &
     "main#" &
     "main_storage#" &
     "max_entry_queue_depth#" &
     "max_entry_queue_length#" &
     "max_queue_length#" &
     "memory_size#" &
     "no_body#" &
     "no_caching#" &
     "no_elaboration_code_all#" &
     "no_inline#" &
     "no_return#" &
     "no_tagged_streams#" &
     "obsolescent#" &
     "optimize#" &
     "ordered#" &
     "pack#" &
     "page#" &
     "part_of#" &
     "passive#" &
     "post#" &
     "postcondition#" &
     "post_class#" &
     "pre#" &
     "precondition#" &
     "predicate#" &
     "predicate_failure#" &
     "preelaborate#" &
     "pre_class#" &
     "provide_shift_operators#" &
     "psect_object#" &
     "pure#" &
     "pure_function#" &
     "refined_depends#" &
     "refined_global#" &
     "refined_post#" &
     "refined_state#" &
     "relative_deadline#" &
     "remote_access_type#" &
     "remote_call_interface#" &
     "remote_types#" &
     "share_generic#" &
     "shared#" &
     "shared_passive#" &
     "simple_storage_pool_type#" &
     "source_reference#" &
     "static_elaboration_desired#" &
     "stream_convert#" &
     "subprogram_variant#" &
     "subtitle#" &
     "suppress_all#" &
     "suppress_debug_info#" &
     "suppress_initialization#" &
     "system_name#" &
     "task_info#" &
     "task_name#" &
     "task_storage#" &
     "test_case#" &
     "thread_local_storage#" &
     "time_slice#" &
     "title#" &
     "type_invariant#" &
     "type_invariant_class#" &
     "unchecked_union#" &
     "unimplemented_unit#" &
     "universal_aliasing#" &
     "unmodified#" &
     "unreferenced#" &
     "unreferenced_objects#" &
     "unreserve_all_interrupts#" &
     "unused#" &
     "volatile#" &
     "volatile_components#" &
     "volatile_full_access#" &
     "volatile_function#" &
     "weak_external#" &
     "ada#" &
     "ada_pass_by_copy#" &
     "ada_pass_by_reference#" &
     "assembler#" &
     "c_variadic_0#" &
     "c_variadic_1#" &
     "c_variadic_2#" &
     "c_variadic_3#" &
     "c_variadic_4#" &
     "c_variadic_5#" &
     "c_variadic_6#" &
     "c_variadic_7#" &
     "c_variadic_8#" &
     "c_variadic_9#" &
     "c_variadic_10#" &
     "c_variadic_11#" &
     "c_variadic_12#" &
     "c_variadic_13#" &
     "c_variadic_14#" &
     "c_variadic_15#" &
     "c_variadic_16#" &
     "cobol#" &
     "cpp#" &
     "fortran#" &
     "intrinsic#" &
     "stdcall#" &
     "stubbed#" &
     "asm#" &
     "assembly#" &
     "default#" &
     "c_plus_plus#" &
     "dll#" &
     "win32#" &
     "address_type#" &
     "aggregate#" &
     "allow#" &
     "amount#" &
     "as_is#" &
     "assertion#" &
     "assertions#" &
     "attribute_name#" &
     "auto#" &
     "body_file_name#" &
     "boolean_entry_barriers#" &
     "by_any#" &
     "by_entry#" &
     "by_protected_procedure#" &
     "casing#" &
     "code#" &
     "component#" &
     "component_size_4#" &
     "copy#" &
     "copy_from#" &
     "copy_to#" &
     "decreases#" &
     "disable#" &
     "dot_replacement#" &
     "dynamic#" &
     "eliminated#" &
     "ensures#" &
     "entity#" &
     "external_name#" &
     "form#" &
     "gcc#" &
     "general#" &
     "gnat#" &
     "gnat_extended_ravenscar#" &
     "gnat_ravenscar_edf#" &
     "gnatprove#" &
     "gpl#" &
     "high_order_first#" &
     "ignore#" &
     "in_out#" &
     "increases#" &
     "info#" &
     "internal#" &
     "ivdep#" &
     "jorvik#" &
     "link_name#" &
     "low_order_first#" &
     "lowercase#" &
     "max_size#" &
     "mechanism#" &
     "message#" &
     "minimized#" &
     "mixedcase#" &
     "mode#" &
     "modified_gpl#" &
     "name#" &
     "new_name#" &
     "no#" &
     "no_dependence#" &
     "no_dynamic_attachment#" &
     "no_dynamic_interrupts#" &
     "no_elaboration_code#" &
     "no_implementation_extensions#" &
     "no_obsolescent_features#" &
     "no_requeue#" &
     "no_requeue_statements#" &
     "no_specification_of_aspect#" &
     "no_task_attributes#" &
     "no_task_attributes_package#" &
     "no_use_of_attribute#" &
     "no_use_of_entity#" &
     "no_use_of_pragma#" &
     "no_unroll#" &
     "no_unrecognized_aspects#" &
     "no_vector#" &
     "nominal#" &
     "none#" &
     "null_address#" &
     "on#" &
     "optional#" &
     "policy#" &
     "parameter_types#" &
     "proof_in#" &
     "reason#" &
     "reference#" &
     "renamed#" &
     "requires#" &
     "restricted#" &
     "result_mechanism#" &
     "result_type#" &
     "robustness#" &
     "runtime#" &
     "sb#" &
     "section#" &
     "semaphore#" &
     "simple_barriers#" &
     "spark#" &
     "spark_05#" &
     "spec_file_name#" &
     "state#" &
     "statement_assertions#" &
     "static#" &
     "stack_size#" &
     "strict#" &
     "structural#" &
     "subunit_file_name#" &
     "suppressible#" &
     "synchronous#" &
     "task_stack_size_default#" &
     "task_type#" &
     "time_slicing_enabled#" &
     "top_guard#" &
     "unit_name#" &
     "unrestricted#" &
     "unroll#" &
     "uppercase#" &
     "user#" &
     "variant#" &
     "vector#" &
     "warn#" &
     "working_storage#" &
     "present#" &
     "abort_signal#" &
     "access#" &
     "address#" &
     "address_size#" &
     "aft#" &
     "alignment#" &
     "asm_input#" &
     "asm_output#" &
     "atomic_always_lock_free#" &
     "bit#" &
     "bit_order#" &
     "bit_position#" &
     "body_version#" &
     "callable#" &
     "caller#" &
     "code_address#" &
     "compiler_version#" &
     "component_size#" &
     "compose#" &
     "constant_indexing#" &
     "constrained#" &
     "count#" &
     "default_bit_order#" &
     "default_scalar_storage_order#" &
     "default_iterator#" &
     "definite#" &
     "delta#" &
     "denorm#" &
     "deref#" &
     "descriptor_size#" &
     "digits#" &
     "elaborated#" &
     "emax#" &
     "enabled#" &
     "enum_rep#" &
     "enum_val#" &
     "epsilon#" &
     "exponent#" &
     "external_tag#" &
     "fast_math#" &
     "finalization_size#" &
     "first#" &
     "first_bit#" &
     "first_valid#" &
     "fixed_value#" &
     "fore#" &
     "has_access_values#" &
     "has_discriminants#" &
     "has_same_storage#" &
     "has_tagged_values#" &
     "identity#" &
     "implicit_dereference#" &
     "index#" &
     "initialized#" &
     "integer_value#" &
     "invalid_value#" &
     "iterator_element#" &
     "iterable#" &
     "large#" &
     "last#" &
     "last_bit#" &
     "last_valid#" &
     "leading_part#" &
     "length#" &
     "library_level#" &
     "loop_entry#" &
     "machine_emax#" &
     "machine_emin#" &
     "machine_mantissa#" &
     "machine_overflows#" &
     "machine_radix#" &
     "machine_rounding#" &
     "machine_rounds#" &
     "machine_size#" &
     "mantissa#" &
     "max_alignment_for_allocation#" &
     "max_integer_size#" &
     "max_size_in_storage_elements#" &
     "maximum_alignment#" &
     "mechanism_code#" &
     "mod#" &
     "model_emin#" &
     "model_epsilon#" &
     "model_mantissa#" &
     "model_small#" &
     "modulus#" &
     "null_parameter#" &
     "object_size#" &
     "old#" &
     "overlaps_storage#" &
     "partition_id#" &
     "passed_by_reference#" &
     "pool_address#" &
     "pos#" &
     "position#" &
     "preelaborable_initialization#" &
     "priority#" &
     "range#" &
     "range_length#" &
     "reduce#" &
     "ref#" &
     "restriction_set#" &
     "result#" &
     "round#" &
     "safe_emax#" &
     "safe_first#" &
     "safe_large#" &
     "safe_last#" &
     "safe_small#" &
     "scalar_storage_order#" &
     "scale#" &
     "scaling#" &
     "signed_zeros#" &
     "size#" &
     "small#" &
     "small_denominator#" &
     "small_numerator#" &
     "storage_size#" &
     "storage_unit#" &
     "stream_size#" &
     "system_allocator_alignment#" &
     "tag#" &
     "target_name#" &
     "terminated#" &
     "to_address#" &
     "type_class#" &
     "type_key#" &
     "unbiased_rounding#" &
     "unchecked_access#" &
     "unconstrained_array#" &
     "universal_literal_string#" &
     "unrestricted_access#" &
     "update#" &
     "vads_size#" &
     "val#" &
     "valid#" &
     "valid_scalars#" &
     "value_size#" &
     "variable_indexing#" &
     "version#" &
     "wchar_t_size#" &
     "wide_wide_width#" &
     "wide_width#" &
     "width#" &
     "word_size#" &
     "adjacent#" &
     "ceiling#" &
     "copy_sign#" &
     "floor#" &
     "fraction#" &
     "from_any#" &
     "image#" &
     "img#" &
     "input#" &
     "machine#" &
     "max#" &
     "min#" &
     "model#" &
     "pred#" &
     "remainder#" &
     "rounding#" &
     "succ#" &
     "to_any#" &
     "truncation#" &
     "typecode#" &
     "valid_value#" &
     "value#" &
     "wide_image#" &
     "wide_wide_image#" &
     "wide_value#" &
     "wide_wide_value#" &
     "output#" &
     "put_image#" &
     "read#" &
     "write#" &
     "elab_body#" &
     "elab_spec#" &
     "elab_subp_body#" &
     "simple_storage_pool#" &
     "storage_pool#" &
     "base#" &
     "class#" &
     "stub_type#" &
     "cpu#" &
     "dispatching_domain#" &
     "interrupt_priority#" &
     "secondary_stack_size#" &
     "ceiling_locking#" &
     "inheritance_locking#" &
     "concurrent_readers_locking#" &
     "fifo_queuing#" &
     "priority_queuing#" &
     "edf_across_priorities#" &
     "fifo_within_priorities#" &
     "non_preemptive_fifo_within_priorities#" &
     "round_robin_within_priorities#" &
     "concurrent#" &
     "sequential#" &
     "short_float#" &
     "float#" &
     "long_float#" &
     "long_long_float#" &
     "signed_8#" &
     "signed_16#" &
     "signed_32#" &
     "signed_64#" &
     "signed_128#" &
     "unsigned_8#" &
     "unsigned_16#" &
     "unsigned_32#" &
     "unsigned_64#" &
     "unsigned_128#" &
     "access_check#" &
     "accessibility_check#" &
     "alignment_check#" &
     "allocation_check#" &
     "atomic_synchronization#" &
     "characters_assertion_check#" &
     "containers_assertion_check#" &
     "discriminant_check#" &
     "division_check#" &
     "duplicated_tag_check#" &
     "elaboration_check#" &
     "index_check#" &
     "interfaces_assertion_check#" &
     "io_assertion_check#" &
     "length_check#" &
     "numerics_assertion_check#" &
     "overflow_check#" &
     "predicate_check#" &
     "program_error_check#" &
     "range_check#" &
     "storage_check#" &
     "strings_assertion_check#" &
     "system_assertion_check#" &
     "tag_check#" &
     "validity_check#" &
     "container_checks#" &
     "tampering_check#" &
     "tasking_check#" &
     "all_checks#" &
     "abort#" &
     "abs#" &
     "accept#" &
     "and#" &
     "all#" &
     "array#" &
     "at#" &
     "begin#" &
     "body#" &
     "case#" &
     "constant#" &
     "declare#" &
     "delay#" &
     "do#" &
     "else#" &
     "elsif#" &
     "end#" &
     "entry#" &
     "exception#" &
     "exit#" &
     "for#" &
     "function#" &
     "generic#" &
     "goto#" &
     "if#" &
     "in#" &
     "is#" &
     "limited#" &
     "loop#" &
     "new#" &
     "not#" &
     "null#" &
     "of#" &
     "or#" &
     "others#" &
     "out#" &
     "package#" &
     "pragma#" &
     "private#" &
     "procedure#" &
     "raise#" &
     "record#" &
     "rem#" &
     "renames#" &
     "return#" &
     "reverse#" &
     "select#" &
     "separate#" &
     "subtype#" &
     "task#" &
     "terminate#" &
     "then#" &
     "type#" &
     "use#" &
     "when#" &
     "while#" &
     "with#" &
     "xor#" &
     "compilation_iso_date#" &
     "compilation_date#" &
     "compilation_time#" &
     "divide#" &
     "enclosing_entity#" &
     "exception_information#" &
     "exception_message#" &
     "exception_name#" &
     "file#" &
     "generic_dispatching_constructor#" &
     "import_address#" &
     "import_largest_value#" &
     "import_value#" &
     "line#" &
     "rotate_left#" &
     "rotate_right#" &
     "shift_left#" &
     "shift_right#" &
     "shift_right_arithmetic#" &
     "source_location#" &
     "unchecked_conversion#" &
     "unchecked_deallocation#" &
     "to_pointer#" &
     "free#" &
     "abstract#" &
     "aliased#" &
     "protected#" &
     "until#" &
     "requeue#" &
     "tagged#" &
     "discriminant#" &
     "operands#" &
     "RESULT#" &
     "synchronous_task_control#" &
     "cursor#" &
     "element#" &
     "has_element#" &
     "forward_iterator#" &
     "reversible_iterator#" &
     "_next#" &
     "previous#" &
     "_previous#" &
     "pseudo_reference#" &
     "reference_control_type#" &
     "get_element_access#" &
     "empty#" &
     "add_named#" &
     "add_unnamed#" &
     "new_indexed#" &
     "assign_indexed#" &
     "interface#" &
     "overriding#" &
     "synchronized#" &
     "some#" &
     "#";

   ---------------------
   -- Generated Names --
   ---------------------

   --  This section lists the various cases of generated names which are
   --  built from existing names by adding unique leading and/or trailing
   --  upper case letters. In some cases these names are built recursively,
   --  in particular names built from types may be built from types which
   --  themselves have generated names. In this list, xxx represents an
   --  existing name to which identifying letters are prepended or appended,
   --  and a trailing n represents a serial number in an external name that
   --  has some semantic significance (e.g. the n'th index type of an array).

   --    xxxA    access type for formal xxx in entry param record   (Exp_Ch9)
   --    xxxB    tag table for tagged type xxx                      (Exp_Ch3)
   --    xxxB    task body procedure for task xxx                   (Exp_Ch9)
   --    xxxD    dispatch table for tagged type xxx                 (Exp_Ch3)
   --    xxxD    discriminal for discriminant xxx                   (Sem_Ch3)
   --    xxxDn   n'th discr check function for rec type xxx         (Exp_Ch3)
   --    xxxE    elaboration boolean flag for task xxx              (Exp_Ch9)
   --    xxxE    dispatch table pointer type for tagged type xxx    (Exp_Ch3)
   --    xxxE    parameters for accept body for entry xxx           (Exp_Ch9)
   --    xxxFn   n'th primitive of a tagged type (named xxx)        (Exp_Ch3)
   --    xxxJ    tag table type index for tagged type xxx           (Exp_Ch3)
   --    xxxM    master Id value for access type xxx                (Exp_Ch3)
   --    xxxP    tag table pointer type for tagged type xxx         (Exp_Ch3)
   --    xxxP    parameter record type for entry xxx                (Exp_Ch9)
   --    xxxPA   access to parameter record type for entry xxx      (Exp_Ch9)
   --    xxxPn   pointer type for n'th primitive of tagged type xxx (Exp_Ch3)
   --    xxxR    dispatch table pointer for tagged type xxx         (Exp_Ch3)
   --    xxxT    tag table type for tagged type xxx                 (Exp_Ch3)
   --    xxxT    literal table for enumeration type xxx             (Sem_Ch3)
   --    xxxV    type for task value record for task xxx            (Exp_Ch9)
   --    xxxX    entry index constant                               (Exp_Ch9)
   --    xxxY    dispatch table type for tagged type xxx            (Exp_Ch3)
   --    xxxZ    size variable for task xxx                         (Exp_Ch9)

   --  TSS names

   --    xxxDA   deep adjust routine for type xxx                   (Exp_TSS)
   --    xxxDF   deep finalize routine for type xxx                 (Exp_TSS)
   --    xxxDI   deep initialize routine for type xxx               (Exp_TSS)
   --    xxxEQ   composite equality routine for record type xxx     (Exp_TSS)
   --    xxxFD   finalize address routine for type xxx              (Exp_TSS)
   --    xxxFA   PolyORB/DSA From_Any converter for type xxx        (Exp_TSS)
   --    xxxIP   initialization procedure for type xxx              (Exp_TSS)
   --    xxxIC   init C++ dispatch tables procedure for type xxx    (Exp_TSS)
   --    xxxRA   RAS type access routine for type xxx               (Exp_TSS)
   --    xxxRD   RAS type dereference routine for type xxx          (Exp_TSS)
   --    xxxRP   Rep to Pos conversion for enumeration type xxx     (Exp_TSS)
   --    xxxSA   array/slice assignment for controlled comp. arrays (Exp_TSS)
   --    xxxSI   stream input attribute subprogram for type xxx     (Exp_TSS)
   --    xxxSO   stream output attribute subprogram for type xxx    (Exp_TSS)
   --    xxxSR   stream read attribute subprogram for type xxx      (Exp_TSS)
   --    xxxSW   stream write attribute subprogram for type xxx     (Exp_TSS)
   --    xxxPI   Put_Image attribute subprogram for type xxx        (Exp_TSS)
   --    xxxTA   PolyORB/DSA To_Any converter for type xxx          (Exp_TSS)
   --    xxxTC   PolyORB/DSA Typecode for type xxx                  (Exp_TSS)

   --  Implicit type names

   --    TxxxT   type of literal table for enumeration type xxx     (Sem_Ch3)

   --  (Note: this list is not complete or accurate ???)

   ----------------------
   -- Get_Attribute_Id --
   ----------------------

   function Get_Attribute_Id (N : Name_Id) return Attribute_Id is
   begin
      if N = Name_CPU then
         return Attribute_CPU;
      elsif N = Name_Dispatching_Domain then
         return Attribute_Dispatching_Domain;
      elsif N = Name_Interrupt_Priority then
         return Attribute_Interrupt_Priority;
      else
         return Attribute_Id'Val (N - First_Attribute_Name);
      end if;
   end Get_Attribute_Id;

   -----------------------
   -- Get_Convention_Id --
   -----------------------

   function Get_Convention_Id (N : Name_Id) return Convention_Id is
   begin
      case N is
         when Name_Ada                   => return Convention_Ada;
         when Name_Ada_Pass_By_Copy      => return Convention_Ada_Pass_By_Copy;
         when Name_Ada_Pass_By_Reference => return
                                              Convention_Ada_Pass_By_Reference;
         when Name_Assembler             => return Convention_Assembler;
         when Name_C                     => return Convention_C;
         when Name_C_Variadic_0          => return Convention_C_Variadic_0;
         when Name_C_Variadic_1          => return Convention_C_Variadic_1;
         when Name_C_Variadic_2          => return Convention_C_Variadic_2;
         when Name_C_Variadic_3          => return Convention_C_Variadic_3;
         when Name_C_Variadic_4          => return Convention_C_Variadic_4;
         when Name_C_Variadic_5          => return Convention_C_Variadic_5;
         when Name_C_Variadic_6          => return Convention_C_Variadic_6;
         when Name_C_Variadic_7          => return Convention_C_Variadic_7;
         when Name_C_Variadic_8          => return Convention_C_Variadic_8;
         when Name_C_Variadic_9          => return Convention_C_Variadic_9;
         when Name_C_Variadic_10         => return Convention_C_Variadic_10;
         when Name_C_Variadic_11         => return Convention_C_Variadic_11;
         when Name_C_Variadic_12         => return Convention_C_Variadic_12;
         when Name_C_Variadic_13         => return Convention_C_Variadic_13;
         when Name_C_Variadic_14         => return Convention_C_Variadic_14;
         when Name_C_Variadic_15         => return Convention_C_Variadic_15;
         when Name_C_Variadic_16         => return Convention_C_Variadic_16;
         when Name_COBOL                 => return Convention_COBOL;
         when Name_CPP                   => return Convention_CPP;
         when Name_Fortran               => return Convention_Fortran;
         when Name_Intrinsic             => return Convention_Intrinsic;
         when Name_Stdcall               => return Convention_Stdcall;
         when Name_Stubbed               => return Convention_Stubbed;

         --  If no direct match, then we must have a convention
         --  identifier pragma that has specified this name.

         when others                     =>
            for J in 1 .. Convention_Identifiers.Last loop
               if N = Convention_Identifiers.Table (J).Name then
                  return Convention_Identifiers.Table (J).Convention;
               end if;
            end loop;

            raise Program_Error;
      end case;
   end Get_Convention_Id;

   -------------------------
   -- Get_Convention_Name --
   -------------------------

   function Get_Convention_Name (C : Convention_Id) return Name_Id is
   begin
      case C is
         when Convention_Ada                   => return Name_Ada;
         when Convention_Ada_Pass_By_Copy      => return Name_Ada_Pass_By_Copy;
         when Convention_Ada_Pass_By_Reference =>
            return Name_Ada_Pass_By_Reference;
         when Convention_Assembler             => return Name_Assembler;
         when Convention_C                     => return Name_C;
         when Convention_C_Variadic_0          => return Name_C_Variadic_0;
         when Convention_C_Variadic_1          => return Name_C_Variadic_1;
         when Convention_C_Variadic_2          => return Name_C_Variadic_2;
         when Convention_C_Variadic_3          => return Name_C_Variadic_3;
         when Convention_C_Variadic_4          => return Name_C_Variadic_4;
         when Convention_C_Variadic_5          => return Name_C_Variadic_5;
         when Convention_C_Variadic_6          => return Name_C_Variadic_6;
         when Convention_C_Variadic_7          => return Name_C_Variadic_7;
         when Convention_C_Variadic_8          => return Name_C_Variadic_8;
         when Convention_C_Variadic_9          => return Name_C_Variadic_9;
         when Convention_C_Variadic_10         => return Name_C_Variadic_10;
         when Convention_C_Variadic_11         => return Name_C_Variadic_11;
         when Convention_C_Variadic_12         => return Name_C_Variadic_12;
         when Convention_C_Variadic_13         => return Name_C_Variadic_13;
         when Convention_C_Variadic_14         => return Name_C_Variadic_14;
         when Convention_C_Variadic_15         => return Name_C_Variadic_15;
         when Convention_C_Variadic_16         => return Name_C_Variadic_16;
         when Convention_COBOL                 => return Name_COBOL;
         when Convention_CPP                   => return Name_CPP;
         when Convention_Entry                 => return Name_Entry;
         when Convention_Fortran               => return Name_Fortran;
         when Convention_Intrinsic             => return Name_Intrinsic;
         when Convention_Protected             => return Name_Protected;
         when Convention_Stdcall               => return Name_Stdcall;
         when Convention_Stubbed               => return Name_Stubbed;
      end case;
   end Get_Convention_Name;

   ---------------------------
   -- Get_Locking_Policy_Id --
   ---------------------------

   function Get_Locking_Policy_Id (N : Name_Id) return Locking_Policy_Id is
   begin
      return Locking_Policy_Id'Val (N - First_Locking_Policy_Name);
   end Get_Locking_Policy_Id;

   -------------------
   -- Get_Pragma_Id --
   -------------------

   function Get_Pragma_Id (N : Name_Id) return Pragma_Id is
   begin
      case N is
         when Name_CPU                              =>
            return Pragma_CPU;
         when Name_Default_Scalar_Storage_Order     =>
            return Pragma_Default_Scalar_Storage_Order;
         when Name_Dispatching_Domain               =>
            return Pragma_Dispatching_Domain;
         when Name_Fast_Math                        =>
            return Pragma_Fast_Math;
         when Name_Interface                        =>
            return Pragma_Interface;
         when Name_Interrupt_Priority               =>
            return Pragma_Interrupt_Priority;
         when Name_Preelaborable_Initialization     =>
            return Pragma_Preelaborable_Initialization;
         when Name_Priority                         =>
            return Pragma_Priority;
         when Name_Secondary_Stack_Size             =>
            return Pragma_Secondary_Stack_Size;
         when Name_Storage_Size                     =>
            return Pragma_Storage_Size;
         when Name_Storage_Unit                     =>
            return Pragma_Storage_Unit;
         when First_Pragma_Name .. Last_Pragma_Name =>
            return Pragma_Id'Val (N - First_Pragma_Name);
         when others                                =>
            return Unknown_Pragma;
      end case;
   end Get_Pragma_Id;

   ---------------------------
   -- Get_Queuing_Policy_Id --
   ---------------------------

   function Get_Queuing_Policy_Id (N : Name_Id) return Queuing_Policy_Id is
   begin
      return Queuing_Policy_Id'Val (N - First_Queuing_Policy_Name);
   end Get_Queuing_Policy_Id;

   ------------------------------------
   -- Get_Task_Dispatching_Policy_Id --
   ------------------------------------

   function Get_Task_Dispatching_Policy_Id
     (N : Name_Id) return Task_Dispatching_Policy_Id
   is
   begin
      return Task_Dispatching_Policy_Id'Val
        (N - First_Task_Dispatching_Policy_Name);
   end Get_Task_Dispatching_Policy_Id;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize is
      P_Index      : Natural;
      Discard_Name : Name_Id;

   begin
      P_Index := Preset_Names'First;
      loop
         Name_Len := 0;
         while Preset_Names (P_Index) /= '#' loop
            Name_Len := Name_Len + 1;
            Name_Buffer (Name_Len) := Preset_Names (P_Index);
            P_Index := P_Index + 1;
         end loop;

         --  We do the Name_Find call to enter the name into the table, but
         --  we don't need to do anything with the result, since we already
         --  initialized all the preset names to have the right value (we
         --  are depending on the order of the names and Preset_Names).

         Discard_Name := Name_Find;
         P_Index := P_Index + 1;
         exit when Preset_Names (P_Index) = '#';
      end loop;

      --  Make sure that number of names in standard table is correct. If this
      --  check fails, run utility program XSNAMES to construct a new properly
      --  matching version of the body.

      pragma Assert (Discard_Name = Last_Predefined_Name);

      --  Initialize the convention identifiers table with the standard set of
      --  synonyms that we recognize for conventions.

      Convention_Identifiers.Init;

      Convention_Identifiers.Append ((Name_Asm,         Convention_Assembler));
      Convention_Identifiers.Append ((Name_Assembly,    Convention_Assembler));

      Convention_Identifiers.Append ((Name_Default,     Convention_C));
      Convention_Identifiers.Append ((Name_External,    Convention_C));

      Convention_Identifiers.Append ((Name_C_Plus_Plus, Convention_CPP));

      Convention_Identifiers.Append ((Name_DLL,         Convention_Stdcall));
      Convention_Identifiers.Append ((Name_Win32,       Convention_Stdcall));
   end Initialize;

   -----------------------
   -- Is_Attribute_Name --
   -----------------------

   function Is_Attribute_Name (N : Name_Id) return Boolean is
   begin
      --  Don't consider Name_Elab_Subp_Body to be a valid attribute name
      --  unless we are working in CodePeer mode.

      return N in First_Attribute_Name .. Last_Attribute_Name
        and then (CodePeer_Mode or else N /= Name_Elab_Subp_Body);
   end Is_Attribute_Name;

   ----------------------------------
   -- Is_Configuration_Pragma_Name --
   ----------------------------------

   function Is_Configuration_Pragma_Name (N : Name_Id) return Boolean is
   begin
      return N in Configuration_Pragma_Names
        or else N = Name_Default_Scalar_Storage_Order
        or else N = Name_Fast_Math;
   end Is_Configuration_Pragma_Name;

   ------------------------
   -- Is_Convention_Name --
   ------------------------

   function Is_Convention_Name (N : Name_Id) return Boolean is
   begin
      --  Check if this is one of the standard conventions

      if N in First_Convention_Name .. Last_Convention_Name
        or else N = Name_C
      then
         return True;

      --  Otherwise check if it is in convention identifier table

      else
         for J in 1 .. Convention_Identifiers.Last loop
            if N = Convention_Identifiers.Table (J).Name then
               return True;
            end if;
         end loop;

         return False;
      end if;
   end Is_Convention_Name;

   ------------------------------
   -- Is_Entity_Attribute_Name --
   ------------------------------

   function Is_Entity_Attribute_Name (N : Name_Id) return Boolean is
   begin
      return N in First_Entity_Attribute_Name .. Last_Entity_Attribute_Name;
   end Is_Entity_Attribute_Name;

   --------------------------------
   -- Is_Function_Attribute_Name --
   --------------------------------

   function Is_Function_Attribute_Name (N : Name_Id) return Boolean is
   begin
      return N in
        First_Renamable_Function_Attribute ..
          Last_Renamable_Function_Attribute;
   end Is_Function_Attribute_Name;

   ---------------------
   -- Is_Keyword_Name --
   ---------------------

   function Is_Keyword_Name (N : Name_Id) return Boolean is
   begin
      return Get_Name_Table_Byte (N) /= 0
        and then (Ada_Version >= Ada_95
                   or else N not in Ada_95_Reserved_Words)
        and then (Ada_Version >= Ada_2005
                   or else N not in Ada_2005_Reserved_Words
                   or else (Debug_Flag_Dot_DD and then N = Name_Overriding))
                   --  Accept 'overriding' keywords if -gnatd.D is used,
                   --  for compatibility with Ada 95 compilers implementing
                   --  only this Ada 2005 extension.
        and then (Ada_Version >= Ada_2012
                   or else N not in Ada_2012_Reserved_Words);
   end Is_Keyword_Name;

   --------------------------------
   -- Is_Internal_Attribute_Name --
   --------------------------------

   function Is_Internal_Attribute_Name (N : Name_Id) return Boolean is
   begin
      return
        N in First_Internal_Attribute_Name .. Last_Internal_Attribute_Name;
   end Is_Internal_Attribute_Name;

   ----------------------------
   -- Is_Locking_Policy_Name --
   ----------------------------

   function Is_Locking_Policy_Name (N : Name_Id) return Boolean is
   begin
      return N in First_Locking_Policy_Name .. Last_Locking_Policy_Name;
   end Is_Locking_Policy_Name;

   ------------------------------------------
   -- Is_Partition_Elaboration_Policy_Name --
   ------------------------------------------

   function Is_Partition_Elaboration_Policy_Name
     (N : Name_Id) return Boolean
   is
   begin
      return N in First_Partition_Elaboration_Policy_Name ..
                  Last_Partition_Elaboration_Policy_Name;
   end Is_Partition_Elaboration_Policy_Name;

   -----------------------------
   -- Is_Operator_Symbol_Name --
   -----------------------------

   function Is_Operator_Symbol_Name (N : Name_Id) return Boolean is
   begin
      return N in First_Operator_Name .. Last_Operator_Name;
   end Is_Operator_Symbol_Name;

   --------------------
   -- Is_Pragma_Name --
   --------------------

   function Is_Pragma_Name (N : Name_Id) return Boolean is
   begin
      return N in First_Pragma_Name .. Last_Pragma_Name
        or else N = Name_CPU
        or else N = Name_Default_Scalar_Storage_Order
        or else N = Name_Dispatching_Domain
        or else N = Name_Fast_Math
        or else N = Name_Interface
        or else N = Name_Interrupt_Priority
        or else N = Name_Preelaborable_Initialization
        or else N = Name_Priority
        or else N = Name_Secondary_Stack_Size
        or else N = Name_Storage_Size
        or else N = Name_Storage_Unit;
   end Is_Pragma_Name;

   ---------------------------------
   -- Is_Procedure_Attribute_Name --
   ---------------------------------

   function Is_Procedure_Attribute_Name (N : Name_Id) return Boolean is
   begin
      return N in First_Procedure_Attribute .. Last_Procedure_Attribute;
   end Is_Procedure_Attribute_Name;

   ----------------------------
   -- Is_Queuing_Policy_Name --
   ----------------------------

   function Is_Queuing_Policy_Name (N : Name_Id) return Boolean is
   begin
      return N in First_Queuing_Policy_Name .. Last_Queuing_Policy_Name;
   end Is_Queuing_Policy_Name;

   -------------------------------------
   -- Is_Task_Dispatching_Policy_Name --
   -------------------------------------

   function Is_Task_Dispatching_Policy_Name (N : Name_Id) return Boolean is
   begin
      return N in First_Task_Dispatching_Policy_Name ..
                  Last_Task_Dispatching_Policy_Name;
   end Is_Task_Dispatching_Policy_Name;

   ----------------------------
   -- Is_Type_Attribute_Name --
   ----------------------------

   function Is_Type_Attribute_Name (N : Name_Id) return Boolean is
   begin
      return N in First_Type_Attribute_Name .. Last_Type_Attribute_Name;
   end Is_Type_Attribute_Name;

   ----------------------------------
   -- Record_Convention_Identifier --
   ----------------------------------

   procedure Record_Convention_Identifier
     (Id         : Name_Id;
      Convention : Convention_Id)
   is
   begin
      Convention_Identifiers.Append ((Id, Convention));
   end Record_Convention_Identifier;

end Snames;
